/*
 * Rollladen.c
 *
 * Created: 19.03.2012 12:10:12
 *  Author: Nenninger
 */ 
//CPU

#define F_CPU 7372800UL 

#include <stdlib.h>
#include <string.h>
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/eeprom.h>
#include <util/delay.h>

#define E0 1
#define E1 2
#define E2 4
#define E3 8
#define E4 16
#define E5 32
#define E6 64
#define E7 128

#define BAUD 115200UL     // Baudrate
#define BAUDMaster 38400UL     // Baudrate

#define DC1     0x11
#define DC2     0x12
#define ACK     0x06
#define NACK    0x15

#define ESC     0x1B
#define CR      0x0D
#define LF      0x0A
#define FF      0x0C
#define BS      0x08
 
#define   EE_DUMMY   	0x000  // Dummyelement (Adresse 0 sollte nicht genutzt werden)
#define   EE_Adr	 	0x001  // Adresse (Stockwerk)
#define   EE_Urlaub	 	0x002  // Urlaub
#define   EE_Automatik	0x003  // Automatik Ein Aus


#define   EE_Helligkeit	0x009  // Helligkeit zum schlieen
#define   EE_HellSonne	0x010  // Helligkeit Sonnenschutz

#define   EEPReadByte(addr)         eeprom_read_byte((uint8_t *)addr)     
#define   EEPWriteByte(addr, val)   eeprom_write_byte((uint8_t *)addr, val) 

// Berechnungen Baudrate
#define UBRR_VAL ((F_CPU+BAUD*8)/(BAUD*16)-1)   // clever runden
#define BAUD_REAL (F_CPU/(16*(UBRR_VAL+1)))     // Reale Baudrate
#define BAUD_ERROR ((BAUD_REAL*1000)/BAUD) // Fehler in Promille, 1000 = kein Fehler.
 
#if ((BAUD_ERROR<990) || (BAUD_ERROR>1010))
  #error Systematischer Fehler der Baudrate nicht genau!!!! 
#endif 

#define UBRR_VALM ((F_CPU+BAUDMaster*8)/(BAUDMaster*16)-1)   // clever runden
#define BAUD_REALM (F_CPU/(16*(UBRR_VALM+1)))     // Reale Baudrate
#define BAUD_ERRORM ((BAUD_REALM*1000)/BAUDMaster) // Fehler in Promille, 1000 = kein Fehler.
 
#if ((BAUD_ERROR<990) || (BAUD_ERROR>1010))
  #error Systematischer Fehler der Baudrate nicht genau!!!! 
#endif 



struct {
	unsigned donline:1; 						//Display Online;
	unsigned monline:1; 						//Master Online;
	unsigned ddopoll:1; 						//Pollen ein aus
	unsigned warteaufmaster:2; 					//Status warte auf Master
	unsigned sensorbefehl:1;
} status;

volatile unsigned int ms; 						//Zhler 1 Sekunde rum
volatile unsigned int dpoll; 					//Zhler polling display
volatile unsigned int dwait; 					//Display max antwort Zeit
volatile unsigned int mwait; 					//Display max antwort Zeit
volatile unsigned int twait; 					//Zeit nach dem die Temperatur gesendet wird

volatile unsigned int bootwait = 30000; 			//Zeit nach dem die Automatik ausgewertet wird

//Rolladen Laufzeiten
volatile unsigned int relaismax[16];			//gesamt Laufzeit Hoch -> runter
volatile unsigned char relaissonne13[8];		//ein drittel der gesamt Laufzeit
volatile unsigned char relaissonne23[8];		//zwei drittel der gesamt Laufzeit
volatile unsigned char relaisoffset;			//Offset welche gesamt Laufzeit benutzt wird

volatile unsigned char relaiswf[8];				//wohl fhl funktion

//Zhler Relais Zeit (sekunden!!!)
volatile unsigned char relaiszeit[8];			//aktulle Zeit vom Startpunkt (oben oder unten) gelaufen

#define rollzeitoffset 1						//Zeit die auf die Laufzeit aufgeschlagen wird
#define WFZ 3									//Zeit die von der MAX Zeit abgezogen wird

/*Letzte funktion auf Relais 
 0= keine Ahnung
 1= luft hoch
 2=	luft runter
 3=	ist oben
 4= ist unten
 5= im hoch laufen gestoppt
 6= im runter laufen gestoppt
 7=	luft in Sonnenposition hoch
 8= luft in Sonnenposition runter
 9= in sonnen position
11= im hoch SS gestopt
12=	im runter SS gestoppt 
*/
volatile unsigned char letztefunktion[8];

volatile unsigned char letztehelligkeit[10];

volatile unsigned char waitstop = 0;
volatile unsigned char waithoch = 0;
volatile unsigned char waitrunter = 0;

struct {
	unsigned stockwerk:1 ; 						//EG=0 OG=1;
	unsigned urlaub:1; 							//0= aus 1= EIN;
	unsigned automatik:1;
	unsigned letztsthell:3; 					//0=wei nicht 1= hoch 2= runter 3= runter Sonne 4= manuelle Steuerung
} optionen;


volatile unsigned char rollnrsz = 0;						//Nummer des Schlafzimmer Rolladen
volatile unsigned char helligkeitswert_soll = 0;			//Helligkeit ab der die Rollen hoch bzw runter gefahren werden
volatile unsigned char helligkeitswert_ist = 70;			//aktulle Helligkeit
volatile unsigned char helligkeitsonne = 0;				//Helligkeit ab der die Sonnenschutz sein soll
volatile unsigned char temperatur_ist = 0;				//
volatile unsigned char maxanzroll = 8;
volatile unsigned char letzteDisplaySeite = 1; //Seite im Display

unsigned char tueren = 11;
unsigned char tor = 0;

//Display
#define dcommaxlen 50 // Maximale Zeichen anzahl

volatile struct {
	unsigned fertig:1; //Daten empfangen (entweder maxstrlen oder Zeichen Timeout erreicht
	unsigned warte:1; // warte auf Zeichen
	unsigned anzahl:6; //Anzahl der Empfangenen Zeichen
       }
dcom;

char dcom_str[dcommaxlen]=""; //Empfangene Zeichen
char dcom_rcv[dcommaxlen]="";	//Empfangene Zeichen zum weiterarbeiten

char dsendstr[150]=""; //Display was senden
unsigned char dsendanz = 0;

volatile unsigned int dcom_rt; //Zhler Zeichen Timeout (RXD)

//UART0 (Display) initialisieren
void dcom_init(void)
{
	UCSR0B = (1<<TXEN0)|(1<<RXEN0)|(1<<RXCIE0);			// UART TX/RX RX Interruppt einschalten
	UCSR0C = (1<<URSEL0) | (1<<UCSZ01) | (1<<UCSZ00);	// Asynchron 8N1 
 
	UBRR0H = UBRR_VAL >> 8;
	UBRR0L = UBRR_VAL & 0xFF;

	dcom.anzahl=0;
	dcom.fertig=0;
}

void ddowaitsendcomplet(void){
	//while (!(UCSR0A & (1<<TXC0))){}
	//loop_until_bit_is_set(UCSR0A, TXC0);
	UCSR0A = (1<<TXC0);
	_delay_us(168);
	PORTD &= ~(1 << PD2); //schalte MAX485 auf empfangen
}

//sende ein Char zum Display
int dsendc(char c)
{
    //while (!(UCSR0A & (1<<UDRE0))){}	
    loop_until_bit_is_set(UCSR0A,UDRE0);/* warten bis Senden moeglich */
	UDR0 = c;                      		/* sende Zeichen */
    return 0;
}

// dem Display einen ganzen String senden (string/anzahl)
void dsend(unsigned char len)
{
	unsigned char count;
	char bcc;
	status.ddopoll = 0;
	dpoll = 0;

	//loop_until_bit_is_set(UCSR0A, TXC0);

	PORTD |= (1 << PD2); //schalte MAX485 auf senden
	//_delay_us(100);
	dsendc(DC1);
	dsendc(len);
	
	bcc=DC1;
	bcc+=len;

	for (count = 0; count < len; count++) //sende die Zeichen
	{
		dsendc(dsendstr[count]);
		bcc+=dsendstr[count];
	}

	dsendc(bcc); //sende BCC
	
	ddowaitsendcomplet();
	status.ddopoll = 1;
	
}

// dem Display "gib deine Daten" senden
void dsendgibdaten(void)
{
   	dpoll = 0; //polling Zhler auf 0 (ist ja polling ;-))
	PORTD |= (1 << PD2); //schalte MAX485 auf senden
	
	if(dsendanz > 0){
		dsend(dsendanz);
		dsendanz =0;
		

	}else{
		dsendc(0x12);
		dsendc(0x1);
		dsendc('S');
		dsendc(0x66);
	}
  
	ddowaitsendcomplet();
}

// dem Display "Adresse 0 aktive" senden
void dsendaktiv(void)
{
	PORTD |= (1 << PD2); //schalte MAX485 auf senden
	
	dsendc(0x12);
	dsendc(0x3);
	dsendc('A');
	dsendc('S');
	dsendc(0x0);
	dsendc(0xa9);
	ddowaitsendcomplet();
}

// dem Display ok senden
void dsendok(void)
{
	PORTD |= (1 << PD2);	//schalte MAX485 auf senden

	dsendc(6);

	ddowaitsendcomplet();
}

//Zeichen von Display empfangen
ISR(USART0_RXC_vect)
{
	unsigned char buffer;
	status.ddopoll = 0;
	dwait = 0;
    dcom_rt = 0;
	
	dpoll = 0;
	buffer = UDR0;							// Daten aus dem Puffer lesen
	if ( dcom.fertig==0 ){					// wenn  gerade nicht in Verarbeitung
		dcom.warte = 1;
		if (dcom.anzahl < dcommaxlen){	// wenn nicht maximal Anzahl
			dcom_str[dcom.anzahl]=buffer;	// Zeichen an den String anhngen
			dcom.anzahl++;					// Anzahl der Zeichen + 1
		} else {
			dcom_str[dcom.anzahl]=buffer;
			//dcom.anzahl=0;
			dcom.fertig=1;
			dcom.warte = 0;
		}
	}
}

void sendstatus(void){
	unsigned char c,rf,pos;

	//\x1bMN\xff\x1bMG\xfd\x64\x1bZL\x64\x00\x64\x00Online\x0D
	memcpy(dsendstr,"\x1bMN\xff\x1bMG\xfd\xff",9);
	rf = 241;
	pos = 9;
	dsendstr[pos] = ESC;
	pos++;
	dsendstr[pos] = 'M';
	pos++;
	dsendstr[pos] = 'N';
	pos++;
	dsendstr[pos] = 249;
	pos++;
	for (c = 0; c < 8; c++) 
	{
	dsendstr[pos] = ESC;
	pos++;
	dsendstr[pos] = 'M';
	pos++;
	dsendstr[pos] = 'K';
	pos++;
	dsendstr[pos] = letztefunktion[c];
	pos++;
	dsendstr[pos] = ESC;
	pos++;
	dsendstr[pos] = 'M';
	pos++;
	dsendstr[pos] = 'N';
	pos++;
	dsendstr[pos] = rf+c;
	pos++;
	}
	//Tren
	for (c = 0; c < 6; c++) {
		dsendstr[pos] = ESC;
		pos++;
		dsendstr[pos] = 'M';
		pos++;
		dsendstr[pos] = 'K';
		pos++;
		if(bit_is_set(tueren,c)>0){
			dsendstr[pos] = 1;
		}else{
			dsendstr[pos] = 0;
		}
		pos++;
		dsendstr[pos] = ESC;
		pos++;
		dsendstr[pos] = 'M';
		pos++;
		dsendstr[pos] = 'N';
		pos++;
		dsendstr[pos] = 231+c;
		pos++;
	}
	//TOR
	dsendstr[pos] = ESC;
	pos++;
	dsendstr[pos] = 'M';
	pos++;
	dsendstr[pos] = 'K';
	pos++;
	dsendstr[pos] = tor;
	pos++;
	dsendstr[pos] = ESC;
	pos++;
	dsendstr[pos] = 'M';
	pos++;
	dsendstr[pos] = 'N';
	pos++;
	dsendstr[pos] = '\xf0';
	pos++;

	dsendanz = pos;
}

void sendoptionen(void){
	unsigned char pos;

	//\x1bMN\xff\x1bMG\xfd\x64\x1bZL\x64\x00\x64\x00Online\x0D
	memcpy(dsendstr,"\x1bMN\xff\x1bMG\xfd\xff",9);
	pos = 9;
	//Urlaub
	dsendstr[pos] = ESC;
	pos++;
	dsendstr[pos] = 'A';
	pos++;
	dsendstr[pos] = 'P';
	pos++;
	dsendstr[pos] = 58+optionen.urlaub;
	pos++;
	dsendstr[pos] = 1;
	pos++;
	//Automatik
	dsendstr[pos] = ESC;
	pos++;
	dsendstr[pos] = 'A';
	pos++;
	dsendstr[pos] = 'P';
	pos++;
	dsendstr[pos] = 67+optionen.automatik;
	pos++;
	dsendstr[pos] = 1;
	pos++;
	//Bars
	dsendstr[pos] = ESC;
	pos++;
	dsendstr[pos] = 'B';
	pos++;
	dsendstr[pos] = 'A';
	pos++;
	dsendstr[pos] = 1;
	pos++;
	dsendstr[pos] = helligkeitswert_soll;
	pos++;
	dsendstr[pos] = ESC;
	pos++;
	dsendstr[pos] = 'B';
	pos++;
	dsendstr[pos] = 'A';
	pos++;
	dsendstr[pos] = 2;
	pos++;
	dsendstr[pos] = helligkeitsonne;
	pos++;
	dsendstr[pos] = ESC;
	pos++;
	dsendstr[pos] = 'B';
	pos++;
	dsendstr[pos] = 'A';
	pos++;
	dsendstr[pos] = 3;
	pos++;
	dsendstr[pos] = helligkeitswert_ist;
	pos++;

	dsendanz = pos;
}

void sendsonne(void){
	unsigned char pos;
	pos = 0;
	dsendstr[pos] = ESC;
	pos++;
	dsendstr[pos] = 'B';
	pos++;
	dsendstr[pos] = 'A';
	pos++;
	dsendstr[pos] = 3;
	pos++;
	dsendstr[pos] = helligkeitswert_ist;
	pos++;
	dsendstr[pos] = ESC;
	pos++;
	dsendstr[pos] = 'M';
	pos++;
	dsendstr[pos] = 'G';
	pos++;
	dsendstr[pos] = 251;
	pos++;
	dsendstr[pos] = 200;
	pos++;
	dsendanz = pos;
}

void sendtemp(void){
	unsigned char pos;
	if(twait==0){
		//\x1bMN\xff\x1bMG\xfd\x64\x1bZL\x64\x00\x64\x00Online\x0D
		//memcpy(dsendstr,"\x1bMN\xff\x1bMG\xfd\xff",9);
		pos = 0;
		//Temperatur
		dsendstr[pos] = ESC;
		pos++;
		dsendstr[pos] = 'B';
		pos++;
		dsendstr[pos] = 'A';
		pos++;
		dsendstr[pos] = 4;
		pos++;
		dsendstr[pos] = temperatur_ist;
		pos++;
		dsendstr[pos] = ESC;
		pos++;
		dsendstr[pos] = 'M';
		pos++;
		dsendstr[pos] = 'G';
		pos++;
		dsendstr[pos] = 251;
		pos++;
		dsendstr[pos] = 200;
		pos++;

		dsendanz = pos;
		twait = 30000;
	}
}

//Master
#define mcommaxlen 50 // Maximale Zeichen anzahl

volatile struct {
	unsigned fertig:1; //Daten empfangen (entweder maxstrlen oder Zeichen Timeout erreicht
	unsigned warte:1; // warte auf Zeichen
	unsigned anzahl:6; //Anzahl der Empfangenen Zeichen
       }
mcom;

char mcom_str[mcommaxlen]=""; //Empfangene Zeichen
char mcom_rcv[mcommaxlen]="";	//Empfangene Zeichen zum weiterarbeiten

volatile unsigned int mcom_rt; //Zhler Zeichen Timeout (RXD)

//UART1 (Master) initialisieren
void mcom_init(void)
{
	UCSR1B = (1<<TXEN1)|(1<<RXEN1)|(1<<RXCIE1);			// UART TX/RX RX Interruppt einschalten
	UCSR1C = (1<<URSEL1) | (1<<UCSZ11) | (1<<UCSZ10);	// Asynchron 8N1 
 
	UBRR1H = UBRR_VALM >> 8;
	UBRR1L = UBRR_VALM & 0xFF;
}

void mdowaitsendcomplet(void){
	//while (!(UCSR1A & (1<<TXC1))){}
	loop_until_bit_is_set(UCSR1A, TXC1);
	UCSR1A = (1<<TXC1);
	_delay_us(250);
	PORTB &= ~(1 << PB4); //schalte MAX485 auf empfangen
}


// sende ein Char zum Master
int msendc(char c)
{
    loop_until_bit_is_set(UCSR1A,UDRE1);	/* warten bis Senden moeglich */
    UDR1 = c;								/* sende Zeichen */
    return 0;
}

// dem Master einen ganzen String senden
void msend(char data[],unsigned char len)
{
	unsigned char count;
	PORTB |= (1 << PB4); //schalte MAX485 auf senden

	for (count = 0; count < len; count++){
		msendc(data[count]);
	}

	mdowaitsendcomplet();
}

// dem Master ein Char senden
void msendnc(char data)
{
	PORTB |= (1 << PB4); //schalte MAX485 auf senden
	msendc(data);
	mdowaitsendcomplet();
}

//Zeichen vom Master Empfangen
ISR(USART1_RXC_vect)
{
	unsigned char buffer;
    mcom_rt = 0;
	
	mwait = 0;
	buffer = UDR1;							// Daten aus dem Puffer lesen
	if ( mcom.fertig==0 ){					// wenn  gerade nicht in Verarbeitung
		mcom.warte = 1;
		if (mcom.anzahl < mcommaxlen){	// wenn nicht maximal Anzahl
			mcom_str[mcom.anzahl]=buffer;	// Zeichen an den String anhngen
			mcom.anzahl++;					// Anzahl der Zeichen + 1
		} else {
			mcom_str[mcom.anzahl]=buffer;
			//dcom.anzahl=0;
			mcom.fertig=1;
			mcom.warte = 0;
		}
	}
}

//Status zum Master
void mantwort(void){
	char cmd[16];
	cmd[0] = '\xfe';
	cmd[1] = optionen.stockwerk;
	cmd[2] = letztefunktion[0];
	cmd[3] = letztefunktion[1];
	cmd[4] = letztefunktion[2];
	cmd[5] = letztefunktion[3];
	cmd[6] = letztefunktion[4];
	cmd[7] = letztefunktion[5];
	cmd[8] = letztefunktion[6];
	cmd[9] = letztefunktion[7];
	cmd[10] = helligkeitswert_ist;
	cmd[11] = optionen.urlaub;
	cmd[12] = helligkeitswert_soll;
	cmd[13] = helligkeitsonne;
	cmd[14] = temperatur_ist;
	cmd[15] = optionen.automatik;
	msend(cmd,16);
}

//Relais funktionen
void statusaenderung(void){
	if(letzteDisplaySeite==3){
		sendstatus();
	}

}

void dohoch(unsigned char r){
	PORTC &= ~(1 << r);
	PORTA |= (1 << r);
	letztefunktion[r] = 1;
	statusaenderung();
}

void hoch(unsigned char r)
{
	//Hoflicht
	if((optionen.stockwerk==0)&(r==7)){
		letztefunktion[7]=0;
	}
	relaiswf[r] = 0;
	switch(letztefunktion[r]){
		case 0: case 4: //unbekannt oder ist unten
			relaiszeit[r] = relaismax[r+relaisoffset];
			dohoch(r);
		break;
		case 2: case 6: //ist gearde am runterfahren +  im runterfahren gestopt
			relaiszeit[r] = (relaismax[r+relaisoffset]-relaiszeit[r])+rollzeitoffset;
			dohoch(r);
		break;
		case 5:
			relaiszeit[r] += rollzeitoffset;
			dohoch(r);		
		break;
		case 7: case 11:
			relaiszeit[r] = (relaismax[r+relaisoffset]-(relaiszeit[r] + relaissonne23[r]))+rollzeitoffset;
			dohoch(r);
		break;		
		case 8: case 12:
			relaiszeit[r] = (relaismax[r+relaisoffset]-(relaiszeit[r] + relaissonne13[r]))+rollzeitoffset;
			dohoch(r);
		break;
		case 9:
			relaiszeit[r] = relaissonne23[r];
			dohoch(r);
		break;
		
		
	}
	
}

void dorunter(unsigned char r){
	PORTA &= ~(1 << r);
	PORTC |= (1 << r);
	letztefunktion[r] = 2;
	statusaenderung();
}

void runter(unsigned char r)
{
	switch(letztefunktion[r]){
		case 0: case 3: //unbekannt oder ist oben
			relaiszeit[r] = relaismax[r+relaisoffset];
			dorunter(r);
		break;
		case 2:			//luft runter
			relaiswf[r] = 1;
		break;
		case 1: case 5:	//ist gearde am hochfahren +  im hochfahren gestopt 
			relaiszeit[r] = (relaismax[r+relaisoffset]-relaiszeit[r])+rollzeitoffset;
			dorunter(r);
		break;
		case 6:
			relaiszeit[r] += rollzeitoffset;
			dorunter(r);
		break;
		case 7: case 11:
			relaiszeit[r] = (relaismax[r+relaisoffset]-(relaiszeit[r] + relaissonne23[r]))+rollzeitoffset;
			dorunter(r);
		break;
		case 8: case 12:
			relaiszeit[r] = (relaismax[r+relaisoffset]-(relaiszeit[r] + relaissonne13[r]))+rollzeitoffset;
			dorunter(r);
		break;
		case 9:
			relaiszeit[r] = relaissonne13[r];
			dorunter(r);
		break;
		
	}
	
}

void stop(unsigned char r)
{
	PORTA &= ~(1 << r);	
	PORTC &= ~(1 << r);
	switch(letztefunktion[r]){
		case 1: case 2: case 7: case 8:
			letztefunktion[r]+=4;
			statusaenderung();
		break;
	}
}

void stopnachzeit(unsigned char r){
	PORTA &= ~(1 << r);	
	PORTC &= ~(1 << r);
	if(letztefunktion[r]<3){
		letztefunktion[r]+=2;
	}
	if((letztefunktion[r]==7)|(letztefunktion[r]==8)){ //Stop in Sonnenposition
		letztefunktion[r]=9;
	}
	statusaenderung();
}

void dosonnehoch(unsigned char r){
	PORTC &= ~(1 << r);
	PORTA |= (1 << r);
	letztefunktion[r] = 7;
	statusaenderung();
}

void dosonnerunter(unsigned char r){
	PORTA &= ~(1 << r);
	PORTC |= (1 << r);
	letztefunktion[r] = 8;
	statusaenderung();
}

void dosonne(unsigned char r){
	switch(letztefunktion[r]){
		case 1: case 5: //luft hoch //im hoch gestoppt
			if(relaiszeit[r] < relaissonne23[r]){
				relaiszeit[r] = (relaissonne23[r]-relaiszeit[r]);
				dosonnerunter(r);
			}else if(relaiszeit[r] > relaissonne23[r]){
				relaiszeit[r] = (relaiszeit[r]-relaissonne23[r]);
				dosonnehoch(r);	
			}
			if(relaiszeit[r] == relaissonne13[r]){
				letztefunktion[r] = 7;
				stopnachzeit(r);
			}
		break;
		case 2: case 6: //luft runter //im runter gestoppt
			if(relaiszeit[r] < relaissonne13[r]){
				relaiszeit[r] = (relaissonne13[r]-relaiszeit[r]);
				dosonnehoch(r);
			}else if(relaiszeit[r] > relaissonne13[r]){
				relaiszeit[r] = (relaiszeit[r]-relaissonne13[r]);
				dosonnerunter(r);	
			}
			if(relaiszeit[r] == relaissonne13[r]){
				letztefunktion[r] = 8;
				stopnachzeit(r);
			}
		break;
		case 3: //ist oben
			relaiszeit[r] = relaissonne23[r];
			dosonnerunter(r);	
		break;
		case 4: //ist unten
			relaiszeit[r] = relaissonne13[r];
			dosonnehoch(r);
		break;
		
		
	}
}

//Timer0
ISR (TIMER0_COMP_vect)
{
	if(dcom.warte == 1){	//warte auf Zeichen
		dcom_rt++;			//dann Zhler Timeout Zeichen +1
	}

	if(mcom.warte == 1){	//warte auf Zeichen
		mcom_rt++;			//dann Zhler Timeout Zeichen +1
	}

	if(dcom_rt == 20){		//Timeout Zeichen (RXD) dann String zum weiterverarbeiten geben
		dcom.warte = 0;
		dcom_rt = 0;
		dcom.fertig=1;
	}

	if(mcom_rt == 20){		//Timeout Zeichen (RXD) dann String zum weiterverarbeiten geben
		mcom.warte = 0;
		mcom_rt = 0;
		mcom.fertig=1;
	}
	
	if(status.ddopoll==1){	//wenn Pollen = ja dann Zhler +1
		dpoll++;
	}

	if(dpoll==50){			//wenn Zeit zum Pollen
		dpoll = 0;
		if(status.donline == 1){	//wenn das Display online ist
			dsendgibdaten();		//dann nur "gibdaten"
		}else{
			dsendaktiv();			//Display aktiv schalten (falls neustart Display)
			//_delay_ms(1000);
			//dsendgibdaten();
		}
	}

  	dwait++;				//Zhler kein Zeichen vom Display (wird im Quelltext manchmal auf 0 gesetzt)
	if(dwait==3000){
		dwait = 0;
		status.donline = 0;	// Display als offline makieren
	}

	mwait++;				//Zhler kein Zeichen vom Display (wird im Quelltext manchmal auf 0 gesetzt)
	if(mwait==3000){
		mwait = 0;
		status.monline = 0;	// Display als offline makieren
	}

	ms++;					//Millisekunden Zhler
	if(ms == 1000){ 		//eine Sekunde rum
		ms = 0;
		unsigned char index;
		for(index = 0; index < 8; index++ ){ //Relais Steuerung
			if((optionen.stockwerk==0)&(index==7)){//wenn Hoflicht dann nicht

			}else{
				switch(letztefunktion[index]){
					case 7: case 8: case 1: case 2:
						if( (relaiswf[index] == 1) & (relaiszeit[index] == WFZ) ){
							relaiswf[index] = 0;
							stop(index);
						}else{
							if(relaiszeit[index]==0){
								stopnachzeit(index);	
							}else{
								relaiszeit[index] = relaiszeit[index] -1;		
							}	
					}		
				break;
				}
			}
		}
	}
	if(waitstop > 0){
		waitstop--;
	}
	if(waithoch > 0){
		waithoch--;
	}
	if(waitrunter > 0){
		waitrunter--;
	}
	if(twait > 0){
		twait--;
	}
	if(bootwait > 0){
		bootwait--;
	}
}


void sichereeinstellungen(unsigned char was){					//Einstellungen in den EEProm schreiben 0=alles 1= nur optionen 2= nur helligkeitswert
	switch(was){
		case 0:
			EEPWriteByte(EE_Adr,optionen.stockwerk);			
		break;
		case 1:
			EEPWriteByte(EE_Urlaub,optionen.urlaub);	
		break;
		case 2:
			EEPWriteByte(EE_Helligkeit,helligkeitswert_soll);	//Helligkeitswert zum schlieen der Rolllden
		break;
		case 3:
			EEPWriteByte(EE_HellSonne,helligkeitsonne);			//Helligkeitswert Sonnenfuntion
		break;
		case 4:
			EEPWriteByte(EE_Automatik,optionen.automatik);	
		break;
	}
		
}

void allerolllaeden(unsigned char f){
    unsigned char c;
	switch(f){
		case 1:
			for(c=0;c<maxanzroll;c++){
				if(c != rollnrsz){
					hoch(c);
					_delay_ms(10);
				}else{
					if(optionen.stockwerk == 0){
						hoch(c);
						_delay_ms(10);
					}
					if(optionen.urlaub==1){
						hoch(c);
						_delay_ms(10);
					}
				}
			}
		break;
		case 2:
			for(c=0;c<maxanzroll;c++){
				if((status.sensorbefehl == 0) | ((status.sensorbefehl == 1) & (letztefunktion[c]!=6))){
					runter(c);
					_delay_ms(10);
				}
					
			}
			status.sensorbefehl = 0;
		break;
		case 3:
			for(c=0;c<maxanzroll;c++){
				stop(c);
			}
		break;
	}
}

void allesonnenschutz(unsigned char f){
	 unsigned char c;
	 if(optionen.stockwerk==1){

		switch(f){
		case 1:
			for(c=4;c<8;c++){
				hoch(c);
				_delay_ms(10);
			}
		break;
		case 2:
			for(c=4;c<8;c++){
				runter(c);
				_delay_ms(10);
			}
		break;
		case 3:
			for(c=4;c<8;c++){
				stop(c);
			}
		break;
		}
	}
}

void sonne(void){
  	dosonne(0);
	_delay_ms(10);
	dosonne(1);
	_delay_ms(10);
	dosonne(2);
	_delay_ms(10);
	dosonne(3);
	if(optionen.stockwerk==0){
		_delay_ms(10);
		dosonne(4);
		_delay_ms(10);
		dosonne(5);
	}

}

void dbefehleauswerten(char len){
	unsigned char r,f,count;
	for (count = 3; count < len; count=count+2){
		r = dcom_rcv[count];						//Wer?
		f = dcom_rcv[count+1];						//Was?

		switch(r){
			case 1: case 2: case 3: case 4: case 5: case 6: case 7: case 8:
				switch(f){								//einzeln
					case 1:
						hoch(r-1);
					break;
					case 2:
						runter(r-1);
					break;
					case 3:
						stop(r-1);
					break;
				}
				
			break;
			case 27:	//Werte vom Display
			  	switch(f){
					case 66: //Scrollbar Wert	
						switch(dcom_rcv[count+3]){
							case 1:		//Helligkeit
								helligkeitswert_soll = dcom_rcv[count+4];
								sichereeinstellungen(2);
							break;
							case 2:		//Sonne
								helligkeitsonne = dcom_rcv[count+4];
								sichereeinstellungen(3);
							break;
						}
					break;
				}				
				count=count+3;				
			break;
			case 200:	// alle Rollden
				allerolllaeden(f);
				
			break;
			case 201: // alle Sonneschutz
				if(f==1){
					if(optionen.stockwerk==1){
				  		allesonnenschutz(2);
					}
					sonne();
				}
			
			break;
			case 250:	//urlaub zurck
				optionen.urlaub = f-1;
				sichereeinstellungen(1);
			break;
			case 251:	//automatik zurck
				optionen.automatik = f-1;
				sichereeinstellungen(4);
			break;
			case 254:
				letzteDisplaySeite = f;
				if(letzteDisplaySeite==1){
					twait = 0;
					sendtemp();
				}
			break;
			case 255:	//Statusseite geffnet
				switch(f){
					case 1: //Statusseite geffnet
						sendstatus();
						
					break;
					case 2: //Optionenseite geffnet
						sendoptionen();
						
					break;
				}
			break;
		}
	}
}

void mbefehleauswerten(char len){
	unsigned char r,f,count;
	for (count = 2; count < len; count=count+2){
		r = mcom_rcv[count];						//Wer?
		f = mcom_rcv[count+1];						//Was?
		switch(r){
			case 0: case 1: case 2: case 3: case 4: case 5: case 6: case 7:
				switch(f){								//einzeln
					case 1:
						hoch(r);
					break;
					case 2:
						runter(r);
					break;
					case 3:
						stop(r);
					break;
				}
				
			break;
			case 200:	// alle Rollden
				allerolllaeden(f);
				
			break;
			case 201: // alle Sonneschutz
				if(f==1){
					if(optionen.stockwerk==1){
				  		allesonnenschutz(2);
					}
					sonne();
				}
				
			break;
			case 250:	//urlaub
				optionen.urlaub = f;
				sichereeinstellungen(1);
			break;
			case 251:	//automatik
				optionen.automatik = f;
				sichereeinstellungen(4);	
			break;
			case 255:
				
			break;
		}
	}
}

//erstmal alles in was definiertes bringen
void init(void) {
	
	/* Beschreibung der ein und Ausgnge
	PA0-7 Rolladen 1-8 hoch
	PC0-7 Rolladen 1-8 runter
	
	PB0=*frei*
	PB1=*frei*
	PB2=Master RS485
	PB3=Master RS485
	PB4=Master RS485 (enabled)
	
	PB5-7 ISP

	PD0=Display RS485
	PD1=Display RS485
	PD2=Display RS485 (enabled)
	
	PD3=*frei*

	PD4=*frei*	
	PD5=*frei*	
	PD6=rote LED Master Offline
	PD7=rote LED Display Offline
	
	PE0=Schlafzimmer hoch
	PE1=Schlafzimmer runter	
	PE2=Schlafzimmer Stop		

	*/

	//Alles als Ausgnge definieren
	DDRA = 0xff;
	DDRB = 0xff;
	DDRC = 0xff;
	DDRD = 0xff;
	DDRE = 0xff;
	
	//Eingnge
	DDRD &= ~ ((1<<PD3));
	DDRE &= ~ ((1<<PE0) | (1<<PE1)| (1<<PE2));

	//interne Pullups
	PORTD |= (1<<PD3);
	PORTE |= (1<<PE0) | (1<<PE1) | (1<<PE2);

	//Ausgnge auf HI
	//PORTB |= (1 << PB2);

	//Ausgnge auf LOW	
	//PORTC &= ~(1 << PC0);
	PORTA = 0x00;
	PORTC = 0x00;

	//ANA_COMP ausschalten;
	ACSR = (1<<ACD);
	
	dcom_init();
	mcom_init();

	// Timer 0 konfigurieren
    TCCR0 |= (1<<CS00) | (1<<CS01) | (1<<WGM01); // Prescaler 64 CTC
	OCR0 = 0x72; // Vergleicher 114 weil 7372800/64 = 115200 
    
	// Compare Interrupt erlauben
	TIMSK |= (1<<OCIE0);
	// Overflow Interrupt
    //TIMSK |= (1<<TOIE0);

	status.ddopoll = 1;
	status.warteaufmaster = 0;
	status.sensorbefehl = 0;
	dcom.warte = 0;
	dwait = 0;
	ms = 0;
	optionen.letztsthell = 0;
	twait = 0;

	//hole Optionen aus EEPROM
	optionen.stockwerk = EEPReadByte(EE_Adr);
	optionen.urlaub = EEPReadByte(EE_Urlaub);
	optionen.automatik = EEPReadByte(EE_Automatik);
	helligkeitswert_soll = EEPReadByte(EE_Helligkeit);		//Helligkeitswert zum schlieen der Rolllden
	helligkeitsonne = EEPReadByte(EE_HellSonne);

	if(optionen.stockwerk == 0){
		rollnrsz = 4;						// wenn EG dann Schlafzimmer = 4
		maxanzroll = 7;						// und nur 7 Rollden
		relaisoffset = 0;					// Relaismax 0-7
	}else{
		rollnrsz = 0;						// wenn OG dann Schlafzimmer = 0
		maxanzroll = 4;						// und nur 4 Rollden
		relaisoffset = 8;					// Relaismax 8-15
	}
	//bei Adresse 0 EG
	relaismax[0] = 19;
	relaismax[1] = 19;
	relaismax[2] = 19;
	relaismax[3] = 19;
	relaismax[4] = 19;
	relaismax[5] = 19;
	relaismax[6] = 15;
	relaismax[7] = 254;
	//bei Adresse 1 OG
	relaismax[8] = 18;
	relaismax[9] = 18;
	relaismax[10] = 18;
	relaismax[11] = 18;
	relaismax[12] = 20;
	relaismax[13] = 20;
	relaismax[14] = 20;
	relaismax[15] = 20;

	unsigned char c;
	// Alles setzen fr Relaissteuerung
	for(c=0;c<8;c++){
		letztefunktion[c] = 0;
		relaiszeit[c] = 0;
		relaissonne13[c] = relaismax[c+relaisoffset]/3;
		relaissonne23[c] = relaissonne13[c]*2;
	}
	//optionen.stockwerk = 0;
	//sichereeinstellungen(0);

	sei();
}

//unbehandelte Interrupt
ISR(BADISR_vect){
	PORTD |= (1 << PD6);
}

//so nun arbeite
int main(void) {
	init();
	while(1) {
		
		if(status.donline == 0){ //LED Display online?
			helligkeitswert_ist = 0;
			PORTD |= (1 << PD7);
		}else{
			PORTD &= ~(1 << PD7);
		}

		if(status.monline == 0){ //LED Master online?
			PORTD |= (1 << PD6);
		}else{
			PORTD &= ~(1 << PD6);
		}
		
		if(dcom.fertig==1){				//Daten vom Display?
			unsigned char len;
			status.ddopoll = 0;			//nicht mehr pollen
			dpoll = 0;
			len = dcom.anzahl;
			dcom.anzahl = 0;
			memcpy(dcom_rcv,dcom_str,len);	//Daten zum weiterarbeiten kopieren
			dcom.fertig=0;

			if(dcom_rcv[len-1]==0x06){  //wenn als letztes ein ACK kam dann
				status.donline = 1;
			}else{
				if((dcom_rcv[0]==ACK) | (dcom_rcv[0]==DC1)){
					_delay_us(100);
					dsendok();				//wir schicken aufjedenfall mal nen ACK
					if(len>4){				//wenn lnger als 3 dann steht was im Telegramm
						dbefehleauswerten(len);
					}

					

				}
			}
			status.ddopoll = 1;
			
		}
		if(mcom.fertig==1){				//Daten vom Master?
			unsigned char len;
			len = mcom.anzahl;
			mcom.anzahl = 0;
			memcpy(mcom_rcv,mcom_str,len);	//Daten zum weiterarbeiten kopieren
			mcom.fertig=0;
			if(mcom_rcv[0]==0xff){								//Polling
				if(mcom_rcv[1]==optionen.stockwerk){			// wenn eigene Adresse 
					status.monline = 1;
					if(len>2){									//wenn lnger als 2 dann steht was im Telegramm
						mbefehleauswerten(len);
					}
					mantwort();
				}
			}
			if(mcom_rcv[0]==0xfe){								//Antwort Slave
				if(mcom_rcv[1]==2){								//Adresse 2 -> Helligkeit und Temperatur
					//Helligkeitsmeldung
					//if(((mcom_rcv[2]<(helligkeitswert_ist-100))|(mcom_rcv[2]>(helligkeitswert_ist+100)))&(helligkeitswert_ist>0)){

					//}else{
						if((mcom_rcv[2]<(helligkeitswert_ist))|(mcom_rcv[2]>(helligkeitswert_ist))){
							unsigned char c;
							uint16_t temphell=0;
							for(c=9;c>0;c--){
								letztehelligkeit[c] = letztehelligkeit[c-1];
								 			
							}
							letztehelligkeit[0] = mcom_rcv[2];
							for(c=0;c<10;c++){
								temphell += letztehelligkeit[c];
								 			
							}
							temphell /= 10;
							helligkeitswert_ist = temphell;
							if(letzteDisplaySeite==4){
								sendsonne();
							}
						}
					//}
					//Temperatur
					if(temperatur_ist!=mcom_rcv[3]){
						temperatur_ist = mcom_rcv[3];
						if(letzteDisplaySeite==1){
							sendtemp();
						}
					}
					if((optionen.automatik==1)&(bootwait==0)){
						//ist es Dunkel?
						if((helligkeitswert_ist < helligkeitswert_soll)&(optionen.letztsthell!=1)){
							if(optionen.urlaub==0){
								unsigned char c;
								for(c=0;c<maxanzroll;c++){
									relaiswf[c] = 1;
									status.sensorbefehl = 1;
								}
							}	
							allerolllaeden(2);
							optionen.letztsthell = 1;
						}
						// ist es Hell?
						if((helligkeitswert_ist > (helligkeitswert_soll+3))&(optionen.letztsthell<2)){
							optionen.letztsthell = 2;
							if(optionen.urlaub==1){	
								allerolllaeden(1);
								hoch(rollnrsz);
							}
						}
						// ist Sonne und ber 21 Grad?
						if((helligkeitswert_ist > helligkeitsonne)&(optionen.letztsthell<3)&(temperatur_ist>128)){
							optionen.letztsthell = 3;
							if(optionen.stockwerk==1){
								allesonnenschutz(2);
							}
							sonne();
						}
					}
				}
				if(mcom_rcv[1]==3){								//Adresse 3 -> Tor
					unsigned char tor_tmp;
					if((mcom_rcv[3]==2)&(mcom_rcv[4]==2)){
						tor_tmp = 1;
					}else{
						if(mcom_rcv[4]!=2){
							tor_tmp = 3;
						}else{
							tor_tmp = 2;
						}
					}
					if((letzteDisplaySeite==3)&(tor!=tor_tmp)){
						tor = tor_tmp;
						sendstatus();
					}
				}
				if(mcom_rcv[1]==4){								//Adresse 4 -> Tren
					if(tueren!=mcom_rcv[2]){
						tueren = mcom_rcv[2];
						if(letzteDisplaySeite==3){
							sendstatus();
						}
					}
				}
				if(mcom_rcv[1]==5){								//Adresse 5 -> GaragenModul
					if(optionen.stockwerk==0){//Hoflicht an wenn Garagen Relais 4 ist an
						if((mcom_rcv[2] & 0x40) == 0x40){
							hoch(7);
						}else{
							stop(7);
						}	
					}
				}
			}
								
		}
		if ((!(( PINE & E0 ) == E0 ))&(waithoch==0)) {							//Schlafzimmer hoch
			waithoch = 100;
			hoch(rollnrsz);
		}
		if ((!(( PINE & E1 ) == E1 ))&(waitrunter==0)) {							//Schlafzimmer runter
			waitrunter = 100;
			runter(rollnrsz);
		}
		if ((!(( PINE & E2 ) == E2 ))&(waitstop==0)) {							//Schlafzimmer stop
			waitstop = 100;
			stop(rollnrsz);
		}
	}
	return 0;
}

